#!/bin/bash
# AWS EC2 LXC Deployment for HDGL Monolith

set -e

# Variables
INSTANCE_TYPE="t3.medium"
AMI_ID="ami-0e872aee57663ae2d"  # Ubuntu 24.04 LTS (us-east-1)
KEY_PAIR="hdgl-key-pair"
SECURITY_GROUP="hdgl-sg"
SUBNET_ID="subnet-0a1b2c3d4e5f67890"
VPC_ID="vpc-0a1b2c3d4e5f67890"

# Create Security Group
aws ec2 create-security-group --group-name $SECURITY_GROUP --description "HDGL LXC SG" --vpc-id $VPC_ID
aws ec2 authorize-security-group-ingress --group-id $(aws ec2 describe-security-groups --filters "Name=group-name,Values=$SECURITY_GROUP" --query 'SecurityGroups[0].GroupId' --output text) \
    --protocol tcp --port 22 --cidr 0.0.0.0/0
aws ec2 authorize-security-group-ingress --group-id $(aws ec2 describe-security-groups --filters "Name=group-name,Values=$SECURITY_GROUP" --query 'SecurityGroups[0].GroupId' --output text) \
    --protocol tcp --port 8080 --cidr 0.0.0.0/0
aws ec2 authorize-security-group-ingress --group-id $(aws ec2 describe-security-groups --filters "Name=group-name,Values=$SECURITY_GROUP" --query 'SecurityGroups[0].GroupId' --output text) \
    --protocol tcp --port 9090 --cidr 0.0.0.0/0

# Launch Aggregator Instance
INSTANCE_ID=$(aws ec2 run-instances \
    --image-id $AMI_ID \
    --count 1 \
    --instance-type $INSTANCE_TYPE \
    --key-name $KEY_PAIR \
    --security-group-ids $SECURITY_GROUP \
    --subnet-id $SUBNET_ID \
    --user-data "#!/bin/bash
    apt-get update && apt-get install -y lxc lxc-templates build-essential libssl-dev libgmp-dev cmake
    lxc-create -t ubuntu -n hdgl-aggregator -- -d ubuntu -r 24.04 -a amd64
    cp hdgl_monolith.c /var/lib/lxc/hdgl-aggregator/rootfs/root/
    lxc-execute -n hdgl-aggregator -- gcc -o /root/hdgl_monolith /root/hdgl_monolith.c -lssl -lcrypto -lgmp -lm
    lxc-start -n hdgl-aggregator -d -- /root/hdgl_monolith --mode=aggregator --port=8080 --max-nodes=10000" \
    --query 'Instances[0].InstanceId' --output text)

echo "Aggregator Instance ID: $INSTANCE_ID"
aws ec2 wait instance-running --instance-ids $INSTANCE_ID

# Launch Node Instances (Auto Scaling)
aws autoscaling create-launch-template \
    --launch-template-name hdgl-node-template \
    --launch-template-data '{
        "ImageId": "'$AMI_ID'",
        "InstanceType": "t3.micro",
        "UserData": "#!/bin/bash\napt-get update && apt-get install -y lxc lxc-templates build-essential libssl-dev libgmp-dev\nlxc-create -t ubuntu -n hdgl-node -- -d ubuntu -r 24.04 -a amd64\ncp hdgl_monolith.c /var/lib/lxc/hdgl-node/rootfs/root/\nlxc-execute -n hdgl-node -- gcc -o /root/hdgl_monolith /root/hdgl_monolith.c -lssl -lcrypto -lgmp -lm\nlxc-start -n hdgl-node -d -- /root/hdgl_monolith --mode=node --aggregator-endpoint=aggregator-ip:8080"
    }'

aws autoscaling create-auto-scaling-group \
    --auto-scaling-group-name hdgl-nodes \
    --launch-template LaunchTemplateName=hdgl-node-template,Version=1 \
    --min-size 1 \
    --max-size 10000 \
    --vpc-zone-identifier $SUBNET_ID

echo "Deployment complete. Aggregator: $INSTANCE_ID, Nodes: hdgl-nodes ASG"